<?php

namespace WPDesk\Library\WPEmail;

use Exception;
use WP_Error;
use WPDesk\Library\WPEmail\Abstracts\EmailInterface;
use WPDesk\Library\WPEmail\Helpers\HTML;
use WPDesk\View\Renderer\Renderer;
use WPDesk\View\Renderer\SimplePhpRenderer;
use WPDesk\View\Resolver\ChainResolver;
use WPDesk\View\Resolver\DirResolver;

class Mailer {

    /**
     * @var EmailInterface[]
     */
    private $emails = [];

    /**
     * @var Renderer
     */
    private $renderer;

    /**
     * @var array
     */
    private $attributes;

    /**
     * @param array $dirs
     */
    public function __construct(
        array $dirs = [],
        array $attributes = []
    ) {
        $this->attributes = $attributes;
        $this->set_renderer( $this->init_renderer( $dirs ) );
    }

    /**
     * @param array $dirs
     *
     * @return Renderer
     */
    private function init_renderer( array $dirs = [] ): Renderer {
        $resolver = new ChainResolver();
        foreach ( $dirs as $dir ) {
            $resolver->appendResolver( new DirResolver( $dir ) );
        }
        $resolver->appendResolver( new DirResolver( dirname( __DIR__ ) . '/templates' ) );

        return new SimplePhpRenderer( $resolver );
    }

    /**
     * @param Renderer $renderer
     *
     * @return void
     */
    public function set_renderer( Renderer $renderer ) {
        $this->renderer = $renderer;
    }

    /**
     * @param EmailInterface $email
     *
     * @return void
     */
    public function add_email( EmailInterface $email ) {
        $this->emails[ $email->get_id() ] = $email;
    }

    /**
     * @return EmailInterface[]
     */
    public function get_emails(): array {
        return $this->emails;
    }

    /** @return void */
    public function send( $include = [] ) {
        foreach ( $this->get_emails() as $email ) {
            if ( ! empty( $include ) && ! in_array( $email->get_id(), $include, true ) ) {
                continue;
            }
            if ( $email->get_is_enable() ) {
                add_filter(
                    'wp_mail_from',
                    $from_cb = static function ( $from ) use ( $email ) {
                        if ( ! empty( $email->get_from() ) ) {
                            return $email->get_from();
                        }

                        return $from;
                    }
                );
                add_filter(
                    'wp_mail_from_name',
                    $from_name_cb = static function ( $from_name ) use ( $email ) {
                        if ( ! empty( $email->get_from_name() ) ) {
                            return $email->get_from_name();
                        }

                        return $from_name;
                    }
                );
                add_action( 'wp_mail_failed', [ $this, 'catch_error' ] );

                try {
                    $success = wp_mail(
                        $email->get_recipients(),
                        $email->get_subject(),
                        $this->get_email_template( $email ),
                        $email->get_headers(),
                        $email->get_attachments()
                    );
                    if ( ! $success ) {
                        throw new MailerException( 'Count not send the mail with wp_mail()' );
                    }
                } catch ( Exception $e ) {
                    if ( $e instanceof MailerException ) {
                        throw $e;
                    }

                    throw new MailerException( sprintf( 'wp_mail() failure. Original error: %s', $e->getMessage() ), 0, $e );
                } finally {
                    remove_action( 'wp_mail_failed', [ $this, 'catch_error' ], 99999 );
                    remove_filter( 'wp_mail_from', $from_cb );
                    remove_filter( 'wp_mail_from_name', $from_name_cb );
                }
            }
        }

    }

    /** @return void */
    public function catch_error( WP_Error $error ) {
        throw MailerException::with_wp_error( $error );
    }

    protected function get_email_template( EmailInterface $email ): string {
        $output = $this->renderer->render(
            'html/email-header',
            [
                'heading' => $this->attributes['heading'] ?? $email->get_heading(),
                'logo'    => $this->attributes['logo']
            ]
        );
        $output .= $this->renderer->render( 'html/' . $email->get_id(), [ 'content' => $email->get_content() ] );
        $output .= $this->renderer->render( 'html/email-footer', [ 'footer' => $this->attributes['footer'] ] );

        return $this->css_inline( $output );
    }

    /**
     * @param string $content
     *
     * @return mixed|string
     */
    protected function css_inline( string $content ): string {
        $styles = $this->renderer->render(
            'html/email-styles',
            [
                'primary' => $this->attributes['primary'] ?? '#d15291',
                'text'    => $this->attributes['text'] ?? '#303030',
                'bg'      => $this->attributes['bg'] ?? '#f9f9f9',
                'body'    => $this->attributes['body'] ?? '#ffffff',
            ]
        );

        return HTML::style_inline( $content, $styles );
    }

}
