<?php

namespace WPDesk\Library\WPEmail\Abstracts;

use Exception;

abstract class EmailAbstract implements EmailInterface {

    /**
     * @var bool
     */
    private $is_enable;

    /**
     * @var array
     */
    protected $recipients = [];

    /**
     * @var array
     */
    protected $placeholders = [];

    /**
     * @var string
     */
    protected $subject = '';

    /**
     * @var string
     */
    protected $heading = '';

    /**
     * @var array
     */
    protected $attachments = [];

    /**
     * @var string
     */
    protected $content = '';

    /**
     * @var string
     */
    protected $type = 'html';

    /**
     * @var array
     */
    protected $headers = [];

    /**
     * Define unique email ID.
     *
     * @return string
     */
    abstract public function get_id(): string;

    public function get_template_name(): string {
        return 'default';
    }

    /**
     * @return string
     */
    public function get_from(): string {
        return '';
    }

    /**
     * @return string
     */
    public function get_from_name(): string {
        return '';
    }

    /**
     * @return bool
     */
    public function get_is_enable(): bool {
        return $this->is_enable;
    }

    /**
     * @param $enable
     *
     * @return EmailAbstract
     */
    public function set_is_enable( $enable ): self {
        $this->is_enable = $enable;

        return $this;
    }

    /**
     * Set placeholders.
     *
     * @param array $placeholders
     *
     * @return self
     */
    public function set_placeholders( array $placeholders = [] ): self {
        $this->placeholders = array_merge( $this->placeholders, $placeholders );

        return $this;
    }

    /**
     * Get defined placeholders.
     *
     * @return string[]
     */
    public function get_placeholders(): array {
        return $this->placeholders;
    }

    /**
     * Get email subject.
     *
     * @param string $subject
     *
     * @return self
     */
    public function set_subject( string $subject ): self {
        $this->subject = $subject;

        return $this;
    }

    /**
     * Get email subject.
     *
     * @return string
     * @throws Exception
     */
    public function get_subject(): string {
        if ( ! $this->subject ) {
            throw new Exception( 'Empty email subject' );
        }

        return $this->replace_placeholders( $this->subject );
    }

    /**
     * Set email heading.
     *
     * @param string $heading
     *
     * @return self
     */
    public function set_heading( string $heading ): self {
        $this->heading = $heading;

        return $this;
    }

    /**
     * Get email heading.
     *
     * @return string
     */
    public function get_heading(): string {
        return $this->replace_placeholders( $this->heading );
    }

    /**
     * Get valid recipients.
     *
     * @param array $recipients
     *
     * @return self
     */
    public function set_recipients( array $recipients = [] ): self {
        $this->recipients = $recipients;

        return $this;
    }

    /**
     * Get valid recipients.
     *
     * @return string[]
     */
    public function get_recipients(): array {
        return $this->recipients;
    }

    /**
     * Get email headers.
     *
     * @param array $headers
     *
     * @return self
     */
    public function set_headers( array $headers = [] ): self {
        $this->headers = $headers;

        return $this;
    }

    /**
     * Get email headers.
     *
     * @return string[]
     */
    public function get_headers(): array {
        return $this->headers;
    }

    /**
     * Set attachments.
     *
     * @param array $attachments
     *
     * @return self
     */
    public function set_attachments( array $attachments ): self {
        $this->attachments = $attachments;

        return $this;
    }

    /**
     * Get email attachments.
     *
     * @return array
     */
    public function get_attachments(): array {
        return $this->attachments;
    }

    /**
     * Set email type.
     *
     * @param string $type
     *
     * @return self
     */
    public function set_type( string $type = 'html' ): self {
        $this->type = $type;

        return $this;
    }

    /**
     * Get email type.
     *
     * @return string
     */
    public function get_type(): string {
        return $this->type;
    }

    /**
     * Get content type.
     *
     * @return string
     */
    public function get_content_type(): string {
        switch ( $this->get_type() ) {
            case 'html':
                return 'text/html';
            case 'multipart':
                return 'multipart/alternative';
            default:
                return 'text/plain';
        }
    }

    /**
     * Get email content.
     *
     * @param string $content
     *
     * @return self
     */
    public function set_content( string $content ): self {
        $this->content = $content;

        return $this;
    }

    /**
     * Get email content.
     *
     * @return string
     * @throws Exception
     */
    public function get_content(): string {
        return $this->replace_placeholders( $this->content );
    }

    /**
     * @param string $string
     *
     * @return array|string|string[]
     */
    protected function replace_placeholders( string $string ): string {
        return (string) str_replace( array_keys( $this->placeholders ), array_values( $this->placeholders ), $string );
    }


}
