<?php

namespace WPDesk\Library\WPEmail;

use Exception;
use WP_Error;
use WPDesk\Library\WPEmail\Abstracts\EmailInterface;
use WPDesk\Library\WPEmail\Helpers\HTML;
use WPDesk\View\Renderer\Renderer;
use WPDesk\View\Renderer\SimplePhpRenderer;
use WPDesk\View\Resolver\ChainResolver;
use WPDesk\View\Resolver\DirResolver;

class Mailer {

    /**
     * @var EmailInterface[]
     */
    private $emails = [];

    /**
     * @var Renderer
     */
    private $renderer;

    /**
     * @var array
     */
    private $template_attributes;

    /**
     * @var string
     */
    private $from;

    /**
     * @var string
     */
    private $from_name;

    /**
     * @param array $dirs
     * @param array $template_attributes
     */
    public function __construct(
        array $dirs = [],
        array $template_attributes = []
    ) {
        $this->template_attributes = $template_attributes;
        $this->set_renderer( $this->init_renderer( $dirs ) );
        $this->set_from( get_bloginfo( 'admin_email' ) );
        $this->set_from_name( get_bloginfo( 'name', 'display' ) );
    }

    /**
     * @param array $dirs
     *
     * @return Renderer
     */
    private function init_renderer( array $dirs = [] ): Renderer {
        $resolver = new ChainResolver();
        foreach ( $dirs as $dir ) {
            $resolver->appendResolver( new DirResolver( $dir ) );
        }
        $resolver->appendResolver( new DirResolver( dirname( __DIR__ ) . '/templates' ) );

        return new SimplePhpRenderer( $resolver );
    }

    /**
     * @param Renderer $renderer
     *
     * @return void
     */
    public function set_renderer( Renderer $renderer ) {
        $this->renderer = $renderer;
    }

    /**
     * @param EmailInterface $email
     *
     * @return void
     */
    public function add_email( EmailInterface $email ) {
        $this->emails[ $email->get_id() ] = $email;
    }

    /**
     * @return EmailInterface[]
     */
    public function get_emails(): array {
        return $this->emails;
    }

    /**
     * @param string $from
     *
     * @return void
     */
    public function set_from( string $from ) {
        $this->from = $from;
    }

    /**
     * @return string
     */
    public function get_from(): string {
        return $this->from;
    }

    /**
     * @param string $from_name
     *
     * @return void
     */
    public function set_from_name( string $from_name ) {
        $this->from_name = $from_name;
    }

    /**
     * @return string
     */
    public function get_from_name(): string {
        return $this->from_name;
    }

    /** @return void */
    public function send( $include = [] ) {
        foreach ( $this->get_emails() as $email ) {
            if ( ! empty( $include ) && ! in_array( $email->get_id(), $include, true ) ) {
                continue;
            }
            if ( $email->get_is_enable() ) {
                $mailer_from = $this->get_from();
                add_filter(
                    'wp_mail_from',
                    $from_cb = static function ( $from ) use ( $mailer_from ) {
                        return $mailer_from;
                    }
                );

                $mailer_from_name = $this->get_from_name();
                add_filter(
                    'wp_mail_from_name',
                    $from_name_cb = static function ( $from_name ) use ( $mailer_from_name ) {
                        return $mailer_from_name;
                    }
                );
                add_action( 'wp_mail_failed', [ $this, 'catch_error' ] );

                try {
                    $success = wp_mail(
                        $email->get_recipients(),
                        $email->get_subject(),
                        $this->get_email_template( $email ),
                        $email->get_headers(),
                        $email->get_attachments()
                    );
                    if ( ! $success ) {
                        throw new MailerException( 'Count not send the mail with wp_mail()' );
                    }
                } catch ( Exception $e ) {
                    if ( $e instanceof MailerException ) {
                        throw $e;
                    }

                    throw new MailerException( sprintf( 'wp_mail() failure. Original error: %s', $e->getMessage() ), 0, $e );
                } finally {
                    remove_action( 'wp_mail_failed', [ $this, 'catch_error' ], 99999 );
                    remove_filter( 'wp_mail_from', $from_cb );
                    remove_filter( 'wp_mail_from_name', $from_name_cb );
                }
            }
        }

    }

    /** @return void */
    public function catch_error( WP_Error $error ) {
        throw MailerException::with_wp_error( $error );
    }

    protected function get_email_template( EmailInterface $email ): string {
        $output = $this->renderer->render(
            'html/email-header',
            [
                'heading' => $this->template_attributes['heading'] ?? $email->get_heading(),
                'logo'    => $this->template_attributes['logo']
            ]
        );
        $output .= $this->renderer->render( 'html/' . $email->get_id(), [ 'content' => $email->get_content() ] );
        $output .= $this->renderer->render( 'html/email-footer', [ 'footer' => $this->template_attributes['footer'] ] );

        return $this->css_inline( $output );
    }

    /**
     * @param string $content
     *
     * @return mixed|string
     */
    protected function css_inline( string $content ): string {
        $styles = $this->renderer->render(
            'html/email-styles',
            [
                'primary' => $this->template_attributes['primary'] ?? '#d15291',
                'text'    => $this->template_attributes['text'] ?? '#303030',
                'bg'      => $this->template_attributes['bg'] ?? '#f9f9f9',
                'body'    => $this->template_attributes['body'] ?? '#ffffff',
            ]
        );

        return HTML::style_inline( $content, $styles );
    }

}
