<?php

namespace WPDesk\Library\WPEmail\Abstracts;

use Exception;

abstract class EmailAbstract implements EmailInterface {

    /**
     * @var bool
     */
    private $is_enable;

    /**
     * @var array
     */
    private $recipients = [];

    /**
     * @var array
     */
    private $placeholders = [];

    /**
     * @var string
     */
    private $subject = '';

    /**
     * @var string
     */
    private $heading = '';

    /**
     * @var array
     */
    private $attachments = [];

    /**
     * @var string
     */
    private $content = '';

    /**
     * @var string
     */
    private $type = 'html';

    /**
     * @var array
     */
    private $headers = [ 'Content-Type: text/html; charset=UTF-8' ];

    /**
     * @var string
     */
    private $from_email;

    /**
     * @var string
     */
    private $from_name;

    /**
     * @return string
     */
    abstract public function get_id(): string;

    /**
     * @return string
     */
    public function get_template_name(): string {
        return 'default';
    }

    /**
     * @return string
     */
    public function get_from(): string {
        return sanitize_email( $this->from_email );
    }

    /**
     * @param string $from_email
     *
     * @return self
     */
    public function set_from( string $from_email ): self {
        $this->from_email = $from_email;

        return $this;
    }

    /**
     * @param string $from_name
     *
     * @return self
     */
    public function set_from_name( string $from_name ): self {
        $this->from_name = $from_name;

        return $this;
    }

    /**
     * @return string
     */
    public function get_from_name(): string {
        return wp_specialchars_decode( esc_html( $this->from_name ), ENT_QUOTES );
    }

    /**
     * @return bool
     */
    public function get_is_enable(): bool {
        return $this->is_enable;
    }

    /**
     * @param $enable
     *
     * @return EmailAbstract
     */
    public function set_is_enable( $enable ): self {
        $this->is_enable = $enable;

        return $this;
    }

    /**
     * @param array $placeholders
     *
     * @return self
     */
    public function set_placeholders( array $placeholders = [] ): self {
        $this->placeholders = array_merge( $this->placeholders, $placeholders );

        return $this;
    }

    /**
     * @return string[]
     */
    public function get_placeholders(): array {
        return $this->placeholders;
    }

    /**
     * @param string $subject
     *
     * @return self
     */
    public function set_subject( string $subject ): self {
        $this->subject = $subject;

        return $this;
    }

    /**
     * @return string
     * @throws Exception
     */
    public function get_subject(): string {
        if ( ! $this->subject ) {
            throw new Exception( 'Empty email subject' );
        }

        return $this->replace_placeholders( $this->subject );
    }

    /**
     * @param string $heading
     *
     * @return self
     */
    public function set_heading( string $heading ): self {
        $this->heading = $heading;

        return $this;
    }

    /**
     * @return string
     */
    public function get_heading(): string {
        return $this->replace_placeholders( $this->heading );
    }

    /**
     * @param array $recipients
     *
     * @return self
     */
    public function set_recipients( array $recipients = [] ): self {
        $this->recipients = $recipients;

        return $this;
    }

    /**
     * @return string[]
     */
    public function get_recipients(): array {
        return $this->recipients;
    }

    /**
     * @param array $headers
     *
     * @return self
     */
    public function set_headers( array $headers = [] ): self {
        $this->headers = $headers;

        return $this;
    }

    /**
     * @return string[]
     */
    public function get_headers(): array {
        return $this->headers;
    }

    /**
     * @param array $attachments
     *
     * @return self
     */
    public function set_attachments( array $attachments ): self {
        $this->attachments = $attachments;

        return $this;
    }

    /**
     * @return array
     */
    public function get_attachments(): array {
        return $this->attachments;
    }

    /**
     * @param string $type
     *
     * @return self
     */
    public function set_type( string $type = 'html' ): self {
        $this->type = $type;

        return $this;
    }

    /**
     * @return string
     */
    public function get_type(): string {
        return $this->type;
    }

    /**
     * @return string
     */
    public function get_content_type(): string {
        switch ( $this->get_type() ) {
            case 'html':
                return 'text/html';
            case 'multipart':
                return 'multipart/alternative';
            default:
                return 'text/plain';
        }
    }

    /**
     * @param string $content
     *
     * @return self
     */
    public function set_content( string $content ): self {
        $this->content = $content;

        return $this;
    }

    /**
     * @return string
     * @throws Exception
     */
    public function get_content(): string {
        return $this->replace_placeholders( $this->content );
    }

    /**
     * @return array|string|string[]
     */
    protected function replace_placeholders( string $string ): string {
        if ( empty( $this->placeholders ) ) {
            return $string;
        }

        return (string) str_replace( array_keys( $this->placeholders ), array_values( $this->placeholders ), $string );
    }


}
